package pl.topteam.security.password.gen;

import static com.google.common.base.Preconditions.checkNotNull;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Map;
import java.util.Random;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.topteam.security.password.PasswordPreferences;

import com.Ostermiller.util.RandPass;
import com.google.common.collect.Range;

public class Utils {
	private static Logger log = LoggerFactory.getLogger(Utils.class);

	private static final String GEN_ALGORITHM = "SHA1PRNG";

	private Random random;
	private RandPass randPass;

	private PasswordPreferences preferences;

	public Utils(@Nonnull PasswordPreferences preferences) {
		this.preferences = checkNotNull(preferences);
		try {
			random = SecureRandom.getInstance(GEN_ALGORITHM);
		} catch (NoSuchAlgorithmException ex) {
			random = new Random();
		}
		randPass = new RandPass();
		Map<CharacterGroup, Integer> groupsOfCharacter = preferences.characterGroups();
		if(groupsOfCharacter != null && !groupsOfCharacter.isEmpty()) {
			for (Map.Entry<CharacterGroup, Integer> chGroup : groupsOfCharacter.entrySet()) {
				char[] charArray = chGroup.getKey() == CharacterGroup.CUSTOM
					? preferences.customChars().toCharArray()
					: chGroup.getKey().getCharacters().toCharArray();
				randPass.addRequirement(charArray, chGroup.getValue());
			}
		}
	}

	/**
	 * @return
	 */
	public String createPassword() {
		int min = preferences.min();
		int length = min + random.nextInt(preferences.max() - min + 1);
		log.debug("password length: " + length);
		return randPass.getPass(length);
	}

	/**
	 * @return group that isn't properly verified or null if pass is OK
	 */
	public PasswordVerificationInfo verifyPassword(@Nonnull String pass) {
		checkNotNull(pass);

		// length
		if(!Range.closed(preferences.min(), preferences.max()).contains(pass.toCharArray().length)) {
			return PasswordVerificationInfo.LENGTH;
		}

		// group of char
		if(preferences.characterGroups() != null) {
			for (Map.Entry<CharacterGroup, Integer> chGroup : preferences.characterGroups().entrySet()) {
				char[] charArray = chGroup.getKey() == CharacterGroup.CUSTOM
					? preferences.customChars().toCharArray()
					: chGroup.getKey().getCharacters().toCharArray();
				
				int numOfChar = 0;
				for(char ch : charArray) {
					numOfChar += StringUtils.countMatches(pass, String.valueOf(ch));
				}
				if(numOfChar < chGroup.getValue()) {
					return Enum.valueOf(PasswordVerificationInfo.class, chGroup.getKey().name());
				}
			}
		}
		return null;
	}
}
