package pl.topteam.dps.db.model;

import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Sets.difference;
import static com.google.common.collect.Sets.intersection;
import static java.lang.String.format;

import java.io.IOException;
import java.io.Reader;
import java.io.StringWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.rules.TemporaryFolder;
import org.mybatis.generator.api.MyBatisGenerator;
import org.mybatis.generator.api.ProgressCallback;
import org.mybatis.generator.config.Configuration;
import org.mybatis.generator.exception.InvalidConfigurationException;
import org.mybatis.generator.internal.DefaultShellCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.topteam.dps.db.model.main.GenerateModelTest;
import pl.topteam.dps.db.util.SchemaUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;

public abstract class AbstractGeneratorTest {
	private static final Logger log = LoggerFactory.getLogger(AbstractGeneratorTest.class);

	@Rule
    public TemporaryFolder tmpWorkingDir = new TemporaryFolder();

	protected List<String> warnings;

	@Before
	public void clearWarnings() {
		if(warnings != null) {
			warnings.clear();
		} else {
			warnings = Lists.newArrayList();
		}
	}

	@After
	public void checkWarnings() {
		filterWarnings();

		// if any other warnings exist -> test fail
		if(!warnings.isEmpty()) {
			log.error(
					Joiner.on("\n").join(warnings));
			throw new RuntimeException("There are warnings for model generation");
		}
	}

	/**
	 * Process given template using properties.
	 *
	 * @param configTemplate
	 * @param properties
	 * @return
	 */
	protected String processConfig(Reader configTemplateReader, Map<String, Object> properties) {
		StringWriter writer = new StringWriter();
		Velocity.evaluate(
				new VelocityContext(properties), writer, GenerateModelTest.class.getCanonicalName(), configTemplateReader);

		return writer.toString();
	}

	/**
	 * Should filter knowing warnings that test shouldn't fail
	 */
	protected void filterWarnings() {
		// ..., 'Property ? exists in root class ?, but does not have a setter.  MyBatis Generator will generate the property.'
		warnings =
				FluentIterable.from(warnings).filter(
						new Predicate<String>() {
							@Override
							public boolean apply(String w) {
								return !w.contains("MyBatis Generator will generate the property");
							}
						}).toList();

		// ..., 'Existing file /mnt/topteam/workspace/dps-model/src/generated/resources/pl/topteam/dps/model/main/MapperConfig.xml was overwritten'
		warnings =
				FluentIterable.from(warnings).filter(
						new Predicate<String>() {
							@Override
							public boolean apply(String w) {
								return !w.contains("was overwritten");
							}
						}).toList();

		// .... 'Root class pl.topteam.utils.stripes.generic.GenericPracownik ...'
		warnings =
				FluentIterable.from(warnings).filter(
						new Predicate<String>() {
							@Override
							public boolean apply(String w) {
								return !w.startsWith("Root class pl.topteam.utils.stripes.generic.GenericPracownik");
							}
						}).toList();
	}

	/**
	 * Generate model classes using Mybatis Generator Plugin and fill warnings if neccessary.
	 *
	 * @param config
	 * @throws SQLException
	 * @throws IOException
	 * @throws InterruptedException
	 * @throws InvalidConfigurationException
	 */
	protected void generateModel(Configuration config) throws SQLException, IOException, InterruptedException, InvalidConfigurationException {
		DefaultShellCallback callback = new DefaultShellCallback(true);

		MyBatisGenerator myBatisGenerator = new MyBatisGenerator(config, callback, warnings);
		myBatisGenerator.generate(new ProgressCallback() {
			@Override
			public void introspectionStarted(int totalTasks) {
				log.info("MBG - introspectionStarted");
			}

			@Override
			public void generationStarted(int totalTasks) {
				log.info("MBG - generationStarted: " + totalTasks);
			}

			@Override
			public void saveStarted(int totalTasks) {
			}

			@Override
			public void startTask(String taskName) {
				log.info(taskName);
			}

			@Override
			public void done() {
				log.info("MBG - done");
			}

			@Override
			public void checkCancel() throws InterruptedException {
			}
		});
	}

	protected void checkGeneratorConfiguration(Connection conn, Set<String> tablesFromConfiguration) throws SQLException {
		// check table synchronization
		Set<String> tablesFromDatabase = ImmutableSet.copyOf(SchemaUtils.getTableNames(conn));

		// różnica
		Set<String> difference =
				new ImmutableSet.Builder<String>()
					.addAll(difference(tablesFromConfiguration, tablesFromDatabase))
					.addAll(difference(tablesFromDatabase, tablesFromConfiguration))
					.build();
		if(!difference.isEmpty()) {
			log.info(format("Tables to check: %s", Joiner.on(",").join(difference)));

			Set<String> intersection = intersection(tablesFromConfiguration, tablesFromDatabase).immutableCopy();

			difference = difference(tablesFromConfiguration, intersection).immutableCopy();
			log.info(format("Remove from configuration: %s",
					(!difference.isEmpty() ? Joiner.on(",").join(difference) : "nothing")));

			difference = difference(tablesFromDatabase, intersection).immutableCopy();
			log.info(format("Add to configuration: %s",
					(!difference.isEmpty() ? Joiner.on(",").join(difference) : "--nothing--")));

			if(!difference.isEmpty()) {
				log.info(format("\n%s",
						Joiner.on("\n").join(
							from(difference)
							.transform(new Function<String, String>() {
								@Override
								public String apply(String tableName) {
									return format("<table tableName=\"%s\">\n</table>", tableName);
								}
							})
						)));
			}

			throw new RuntimeException("Check configuration file for differences config<->database!");
		}
	}
}
