package pl.topteam.dps.db.model.main;

import static pl.topteam.dps.db.generator.SQLExecutor.executeSql;

import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Set;

import org.h2.jdbcx.JdbcConnectionPool;
import org.junit.Test;
import org.mybatis.generator.config.Configuration;
import org.mybatis.generator.config.Context;
import org.mybatis.generator.config.TableConfiguration;
import org.mybatis.generator.config.xml.ConfigurationParser;
import org.mybatis.generator.exception.InvalidConfigurationException;
import org.mybatis.generator.exception.XMLParserException;

import pl.topteam.dps.db.model.AbstractGeneratorTest;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import com.google.common.io.CharStreams;

public class GenerateModelTest extends AbstractGeneratorTest {
	/**
	 * Porównuje konfigurację głównej bazy danych (tabele) i konfigurację generatora.
	 * Baza dziennika nie jest porównywana z uwagi na stosunkową jej niezmienność (póki co).
	 *
	 * @throws IOException
	 * @throws XMLParserException
	 * @throws InvalidConfigurationException
	 * @throws SQLException
	 * @throws InterruptedException
	 */
	@Test
	public void generate() throws IOException, XMLParserException,
			InvalidConfigurationException, SQLException, InterruptedException {
		// create main database in-memory creation
		String mainJdbcURL = "jdbc:h2:mem:test_main";
		JdbcConnectionPool jdbccpMain = JdbcConnectionPool.create(mainJdbcURL, "sysdba", "masterkey");
		try (Connection conn = jdbccpMain.getConnection()) {
			for (Path[] s : pl.topteam.dps.db.generator.main.CreateDatabaseTest.schemaScripts()) {
				executeSql(conn, s[0].toString(), CharStreams.toString(new InputStreamReader(getClass().getResourceAsStream("/" + s[0]), Charsets.UTF_8)));
			}
		}

		// create dziennik database
		String dziennikJdbcURL = "jdbc:h2:mem:test_dziennik";
		JdbcConnectionPool jdbccpDziennik = JdbcConnectionPool.create(dziennikJdbcURL, "sysdba", "masterkey");
		try (Connection conn = jdbccpDziennik.getConnection()) {
			for (Path[] s : pl.topteam.dps.db.generator.dziennik.CreateDatabaseTest.schemaScripts()) {
				executeSql(conn, s[0].toString(), CharStreams.toString(new InputStreamReader(getClass().getResourceAsStream("/" + s[0]), Charsets.UTF_8)));
			}
		}

		// temporary generation dir
		File tempGeneratedDir = tmpWorkingDir.newFolder("_mybatis-generator");

		// generator parameter
		ImmutableMap<String, Object> properties = new ImmutableMap.Builder<String, Object>()
				.put("_mybatis_generator_tmp", tempGeneratedDir.getAbsolutePath())

				.put("main_jdbcDriver", "org.h2.Driver")
				.put("main_jdbcUserId", "sysdba")
				.put("main_jdbcPassword", "masterkey")
				.put("main_jdbcURL", mainJdbcURL)

				.put("dziennik_jdbcDriver", "org.h2.Driver")
				.put("dziennik_jdbcUserId", "sysdba")
				.put("dziennik_jdbcPassword", "masterkey")
				.put("dziennik_jdbcURL", dziennikJdbcURL)
				.build();

		// generator parser and final configuration
		ConfigurationParser cp = new ConfigurationParser(warnings);
		Configuration config =
				cp.parseConfiguration(new StringReader(processConfig(new InputStreamReader(
						getClass().getResourceAsStream("/mybatis-DPS-config.xml"), Charsets.UTF_8), properties)));

		// check table synchronization
		Set<String> tablesFromConfiguration = Sets.newHashSet();
		for(Context c : config.getContexts()) {
			switch(c.getId()) {
				case "mainDpsModel":
				case "mainDpsPracownikModel":
				case "mainDpsChangelogModel":
					tablesFromConfiguration.addAll(
							FluentIterable
								.from(c.getTableConfigurations())
								.transform(new Function<TableConfiguration, String>() {
									@Override
									public String apply(TableConfiguration t) {
										return t.getTableName();
									}
								}).toList());
					break;
			}
		}
		try (Connection conn = jdbccpMain.getConnection()) {
			checkGeneratorConfiguration(conn, tablesFromConfiguration);
		}

		// GENERATE yeah :)
		generateModel(config);
	}
}
