package pl.topteam.otm.utils;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlEnum;
import javax.xml.bind.annotation.XmlType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;

/**
 * ROZSZERZONA KOPIA z projektu otm o expand kolekcji
 * @author michal
 * @author grzegorz
 */
public class Processor {
	private static final Logger LOGGER = LoggerFactory.getLogger(Processor.class);

	private static final Class<?>[] CLASSES = {
		pl.gov.du.r2021.poz893.wywiad.cz1i2.ObjectFactory.class,
		pl.gov.du.r2021.poz893.wywiad.cz3i4.ObjectFactory.class,
		pl.gov.du.r2021.poz893.wywiad.cz5.ObjectFactory.class,
		pl.gov.du.r2021.poz893.wywiad.cz6.ObjectFactory.class,
		pl.gov.du.r2021.poz893.wywiad.cz7.ObjectFactory.class,
		pl.gov.du.r2021.poz893.wywiad.cz8.ObjectFactory.class,
		pl.gov.du.r2021.poz893.wywiad.cz9.ObjectFactory.class,

		pl.gov.du.r2021.poz893.wywiad.wspolne.ObjectFactory.class,
	};

	private static final Map<Class<?>, Supplier<?>> SUPPLIERS;
	static {
		Builder<Class<?>, Supplier<?>> suppliers = ImmutableMap.builder();
		try {
			for (Class<?> clazz : CLASSES) {
				Object factory = clazz.getConstructor().newInstance();
				for (Method method : clazz.getDeclaredMethods()) {
					if (method.getParameterCount() == 0) {
						Class<?> type = method.getReturnType();
						suppliers.put(type, new Invoker<>(factory, method));
					}
				}
			}
		} catch (Exception e) {
			throw new Error(e);
		}
		SUPPLIERS = suppliers.build();
	}

	public static <T> T expand(T object) {
		if (!isXmlComplexType(object)) {
			return object;
		}
		try {
			Class<?> clazz = object.getClass();
			BeanInfo info = Introspector.getBeanInfo(clazz);
			for (PropertyDescriptor descriptor : info.getPropertyDescriptors()) {
				if (Collection.class.isAssignableFrom(descriptor.getPropertyType())) {
					Object collection = descriptor.getReadMethod().invoke(object);
					for (Object value : (Collection<?>) collection) {
						expand(value);
					}
				} else {
					Supplier<?> supplier = SUPPLIERS.get(descriptor.getPropertyType());
					if (supplier != null) {
						Object value = descriptor.getReadMethod().invoke(object);
						descriptor.getWriteMethod().invoke(object, expand(value != null ? value : supplier.get()));
					}
				}
			}
			return object;
		} catch (Exception e) {
			LOGGER.error("", e);
			throw new RuntimeException(e);
		}
	}

	// nie jest symetryczny względem expand, brak uwzględniania kolekcji
	public static <T> T collapse(T object) {
		if (!isXmlComplexType(object)) {
			return object;
		}
		try {
			boolean isEmpty = true;
			for (Class<?> clazz = object.getClass(); clazz != null; clazz = clazz.getSuperclass()) {
				for (Field field : clazz.getDeclaredFields()) {
					if (isXmlElement(field)) {
						field.setAccessible(true);
						Object value = collapse(field.get(object));
						isEmpty &= isEmpty(value);
						field.set(object, value);
						field.setAccessible(false);
					}
				}
			}
			return !isEmpty ? object : null;
		} catch (Exception e) {
			LOGGER.error("", e);
			throw new RuntimeException(e);
		}
	}

	// helpers

	private static <T> boolean isXmlComplexType(T object) {
		return isXmlType(object) && !isXmlEnum(object);
	}

	private static boolean isXmlType(Object object) {
		return object != null && object.getClass().isAnnotationPresent(XmlType.class);
	}

	private static boolean isXmlEnum(Object object) {
		return object != null && object.getClass().isAnnotationPresent(XmlEnum.class);
	}

	private static boolean isXmlElement(Field field) {
		return field != null && field.isAnnotationPresent(XmlElement.class);
	}

	private static boolean isEmpty(Object value) {
		if (value instanceof List) {
			List<?> list = (List<?>) value;
			return list.isEmpty();
		} else if (value instanceof String) {
			String string = (String) value;
			return string.isEmpty();
		} else {
			return value == null;
		}
	}

	private static final class Invoker<T> implements Supplier<T> {
		private static final Logger LOGGER = LoggerFactory.getLogger(Invoker.class);
		//
		private final Object instance;
		private final Method method;

		public Invoker(Object instance, Method method) {
			this.instance = instance;
			this.method = method;
		}

		@Override
		public T get() {
			try {
				@SuppressWarnings("unchecked")
				T value = (T) method.invoke(instance);
				return value;
			} catch (Exception e) {
				LOGGER.error("", e);
				throw new RuntimeException(e);
			}
		}

		@Override
		public String toString() {
			return method.toString();
		}
	}
}
