package pl.gov.mpips.slownikicentralne;

import static pl.gov.mpips.slownikicentralne.model.Pozycja.pozycja;
import static pl.gov.mpips.slownikicentralne.model.Slownik.slownik;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

import javax.annotation.Nonnull;
import javax.lang.model.element.Modifier;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.io.MoreFiles;
import com.ibm.icu.text.Transliterator;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeSpec;

import pl.gov.mpips.slownikicentralne.model.Pozycja;
import pl.gov.mpips.slownikicentralne.model.Slownik;
import pl.topteam.common.util.ExtraLocales;

public final class GeneratorSlownikow {
	private final static Transliterator transliterator = Transliterator.getInstance("Latin-ASCII");
	
	private GeneratorSlownikow() {
	}
	
	public static void main(String[] args) throws IOException {
		Path wejscie = Paths.get(args[0]);
		Path wyjscie = Paths.get(args[1]);
		ListMultimap<Slownik, Pozycja> slowniki = slowniki(wejscie);
		for (Slownik slownik : slowniki.keySet()) {
			TypeSpec.Builder enumeracja = TypeSpec.enumBuilder(nazwa(slownik)).addModifiers(Modifier.PUBLIC)
				.addJavadoc(slownik.opis() + " " + "[" + slownik.kod() + "]");
			for (Pozycja pozycja : slowniki.get(slownik)) {
				enumeracja.addEnumConstant(nazwa(pozycja), TypeSpec.anonymousClassBuilder("$S, $S", pozycja.kod(), pozycja.opis())
					.addJavadoc(pozycja.opis() + " " + "[" + pozycja.kod() + "]").build());
			}
			enumeracja.addField(FieldSpec.builder(String.class, "kod", Modifier.PRIVATE, Modifier.FINAL).addAnnotation(Nonnull.class).build());
			enumeracja.addField(FieldSpec.builder(String.class, "opis", Modifier.PRIVATE, Modifier.FINAL).addAnnotation(Nonnull.class).build());
			MethodSpec.Builder konstruktor = MethodSpec.constructorBuilder();
			konstruktor.addParameter(String.class, "kod");
			konstruktor.addParameter(String.class, "opis");
			for (String parametr : ImmutableList.of("kod", "opis")) {
				konstruktor.addStatement("this.$L = $L", parametr, parametr);
			}
			enumeracja.addMethod(konstruktor.build());
			enumeracja.addMethod(MethodSpec.methodBuilder("kod").addModifiers(Modifier.PUBLIC).returns(String.class).addStatement("return $L", "kod").build());
			enumeracja.addMethod(MethodSpec.methodBuilder("opis").addModifiers(Modifier.PUBLIC).returns(String.class).addStatement("return $L", "opis").build());
			JavaFile.builder("pl.gov.mpips.slownikicentralne", enumeracja.build()).build().writeTo(wyjscie);
		}
	}
	
	private static String nazwa(Slownik slownik) {
		return identyfikator(transliterator.transliterate(slownik.opis()));
	}
	
	private static String nazwa(Pozycja pozycja) {
		return identyfikator(transliterator.transliterate(pozycja.opis())).toUpperCase(ExtraLocales.PL);
	}
	
	private static String identyfikator(String nazwa) {
		StringBuilder identyfikator = new StringBuilder();
		for (int kod : nazwa.codePoints().toArray()) {
			if (Character.isJavaIdentifierPart(kod)) {
				identyfikator.appendCodePoint(kod);
			} else {
				identyfikator.append("_");
			}
		}
		return identyfikator.toString();
	}
	
	private static ImmutableListMultimap<Slownik, Pozycja> slowniki(Path katalog) throws IOException {
		ImmutableListMultimap.Builder<Slownik, Pozycja> slowniki = ImmutableListMultimap.builder();
		for (Path plik : MoreFiles.listFiles(katalog)) {
			if (MoreFiles.getFileExtension(plik).equals("txt")) {
				Path txt = katalog.resolve(MoreFiles.getNameWithoutExtension(plik) + ".txt");
				Path csv = katalog.resolve(MoreFiles.getNameWithoutExtension(plik) + ".csv");
				Slownik slownik = slownik(
					MoreFiles.getNameWithoutExtension(txt),
					MoreFiles.asCharSource(txt, StandardCharsets.UTF_8).readFirstLine()
				);
				for (String rekord : MoreFiles.asCharSource(csv, StandardCharsets.UTF_8).readLines()) {
					List<String> pola = Splitter.on(";").limit(2).splitToList(rekord);
					Pozycja pozycja = pozycja(pola.get(0), pola.get(1));
					slowniki.put(slownik, pozycja);
				}
			}
		}
		return slowniki.build();
	}
}
