package pl.gov.mpips.zbc.v20200306;

import java.util.List;
import java.util.Objects;
import java.util.function.IntPredicate;
import java.util.function.Predicate;

import pl.gov.mpips.zbc.v20200306.SwiadczenieSprawozdawcze.CzlonekRodziny;
import pl.topteam.pomost.integracja.zbc.BasicErrors;
import pl.topteam.pomost.integracja.zbc.SimpleValidator;

public class W09Validator implements SimpleValidator<SwiadczenieSprawozdawcze> {
	@Override
	public void validate(SwiadczenieSprawozdawcze target, BasicErrors errors) {
		if (weryfikowalne(target)) {
			SytuacjaRodziny sytuacjaRodziny = target.getSytuacjaRodziny();
			String kodSkladuRodziny = sytuacjaRodziny.getKodSkladuRodziny();
			if (!predykat(kodSkladuRodziny).test(target)) {
				errors.rejectValue("sytuacjaRodziny.kodSkladuRodziny", "W09", "Wartość kodu składu rodziny jest niezgodna ze składem rodziny");
			}
		}
	}
	
	private boolean weryfikowalne(SwiadczenieSprawozdawcze swiadczenieSprawozdawcze) {
		List<CzlonekRodziny> czlonkowieRodziny = swiadczenieSprawozdawcze.getCzlonkowieRodziny();
		if (czlonkowieRodziny == null) return false; // ??
		SytuacjaRodziny sytuacjaRodziny = swiadczenieSprawozdawcze.getSytuacjaRodziny();
		if (sytuacjaRodziny == null) return false; // ??
		String kodSkladuRodziny = sytuacjaRodziny.getKodSkladuRodziny();
		if (kodSkladuRodziny == null) return false; // W01
		if (sytuacjaRodziny.getWielkosc() == null) return false; // W01
		if (czlonkowieRodziny.size() != sytuacjaRodziny.getWielkosc()) return false; // W22
		if (czlonkowieRodziny.stream().map(CzlonekRodziny::getSytuacjaOsoby).anyMatch(Objects::isNull)) return false; // ??
		return true;
	}
	
	private Predicate<SwiadczenieSprawozdawcze> predykat(String kodSkladuRodziny) {
		switch (kodSkladuRodziny) {
			// małżeństwo
			case "110": return malzenstwo(d -> d == 0);
			case "121": return malzenstwo(d -> d == 1);
			case "122": return malzenstwo(d -> d == 2);
			case "123": return malzenstwo(d -> d == 3);
			case "124": return malzenstwo(d -> d == 4);
			case "125": return malzenstwo(d -> d == 5);
			case "126": return malzenstwo(d -> d == 6);
			case "127": return malzenstwo(d -> d >= 7);
			// rodzina niepełna
			case "131": return samotnaMatka(d -> d == 1);
			case "132": return samotnaMatka(d -> d == 2);
			case "133": return samotnaMatka(d -> d == 3);
			case "134": return samotnaMatka(d -> d >= 4);
			case "135": return samotnyOjciec(d -> d == 1);
			case "136": return samotnyOjciec(d -> d == 2);
			case "137": return samotnyOjciec(d -> d == 3);
			case "138": return samotnyOjciec(d -> d >= 4);
			// konkubinat
			case "140": return konkubinat(d -> d == 0);
			case "151": return konkubinat(d -> d == 1);
			case "152": return konkubinat(d -> d == 2);
			case "153": return konkubinat(d -> d == 3);
			case "154": return konkubinat(d -> d == 4);
			case "155": return konkubinat(d -> d == 5);
			case "156": return konkubinat(d -> d == 6);
			case "157": return konkubinat(d -> d >= 7);
			// dwurodzinne spokrewnione w linii prostej
			case "211": return wielorodzinne(2);
			// dwurodzinne inne
			// XXX tak jak w walidatorze krzyżowym
			case "219": return konkubinat(d -> d >= 1);
			// trzyrodzinne spokrewnione w linii prostej
			case "221": return wielorodzinne(3);
			// trzyrodzinne inne
			// XXX inaczej niż w walidatorze krzyżowym
			case "229": return konkubinat(d -> d >= 2);
			// cztero i więcej rodzinne
			case "230": return wielorodzinne(4);
			// nierodzinne
			case "310": return dowolnySklad(o -> o == 1);
			case "320": return dowolnySklad(o -> o >= 2)
				// XXX TP-51
				.and(malzenstwo(d -> d == 0).negate())
				.and(konkubinat(d -> d == 0).negate());
			// nieuwzględnione
			default: return dowolnySklad(o -> o >= 1);
		}
	}
	
	// predykaty podstawowe
	
	private Predicate<SwiadczenieSprawozdawcze> malzenstwo(IntPredicate liczbaDzieci) {
		Predicate<CzlonekRodziny> _ojciec = ZONATY.and(MEZCZYZNA).and(POKREWIENSTWO_NIEOKRESLONE.or(MAZ));
		Predicate<CzlonekRodziny> _matka = ZAMEZNA.and(KOBIETA).and(POKREWIENSTWO_NIEOKRESLONE.or(ZONA));
		return rodzice(_ojciec, _matka, liczbaDzieci);
	}
	
	private Predicate<SwiadczenieSprawozdawcze> konkubinat(IntPredicate liczbaDzieci) {
		Predicate<CzlonekRodziny> _ojciec = KONKUBENT.and(MEZCZYZNA).and(POKREWIENSTWO_NIEOKRESLONE.or(BRAK_POKREWIENSTWA));
		Predicate<CzlonekRodziny> _matka = KONKUBINA.and(KOBIETA).and(POKREWIENSTWO_NIEOKRESLONE.or(BRAK_POKREWIENSTWA));
		return rodzice(_ojciec, _matka, liczbaDzieci);
	}
	
	private Predicate<SwiadczenieSprawozdawcze> samotnaMatka(IntPredicate liczbaDzieci) {
		return rodzic(KOBIETA.and(POKREWIENSTWO_NIEOKRESLONE), liczbaDzieci);
	}
	
	private Predicate<SwiadczenieSprawozdawcze> samotnyOjciec(IntPredicate liczbaDzieci) {
		return rodzic(MEZCZYZNA.and(POKREWIENSTWO_NIEOKRESLONE), liczbaDzieci);
	}
	
	private Predicate<SwiadczenieSprawozdawcze> samotnyRodzic(IntPredicate liczbaDzieci) {
		return rodzic(POKREWIENSTWO_NIEOKRESLONE, liczbaDzieci);
	}
	
	private Predicate<SwiadczenieSprawozdawcze> wielorodzinne(int r) {
		assert r >= 2;
		return samotnyRodzic(d -> d >= Math.max(r - 1, 1))
			.or(malzenstwo(d -> d >= Math.max(r - 1, 1)))
			.or(konkubinat(d -> d >= Math.max(r - 2, 1)));
	}
	
	private Predicate<SwiadczenieSprawozdawcze> rodzice(Predicate<CzlonekRodziny> _rodzic1, Predicate<CzlonekRodziny> _rodzic2, IntPredicate liczbaDzieci) {
		return swiadczenieSprawozdawcze -> {
			List<CzlonekRodziny> czlonkowieRodziny = swiadczenieSprawozdawcze.getCzlonkowieRodziny();
			int glowa = (int) czlonkowieRodziny.stream().filter(POKREWIENSTWO_NIEOKRESLONE).count();
			if (glowa != 1) return false;
			int rodzic1 = (int) czlonkowieRodziny.stream().filter(_rodzic1).count();
			if (rodzic1 != 1) return false;
			int rodzic2 = (int) czlonkowieRodziny.stream().filter(_rodzic2).count();
			if (rodzic2 != 1) return false;
			int dzieci = dzieci(czlonkowieRodziny);
			if (rodzic1 + rodzic2 + dzieci != czlonkowieRodziny.size()) return false;
			return liczbaDzieci.test(dzieci);
		};
	}
	
	private Predicate<SwiadczenieSprawozdawcze> rodzic(Predicate<CzlonekRodziny> _rodzic, IntPredicate liczbaDzieci) {
		return swiadczenieSprawozdawcze -> {
			List<CzlonekRodziny> czlonkowieRodziny = swiadczenieSprawozdawcze.getCzlonkowieRodziny();
			int glowa = (int) czlonkowieRodziny.stream().filter(POKREWIENSTWO_NIEOKRESLONE).count();
			if (glowa != 1) return false;
			int rodzic = (int) czlonkowieRodziny.stream().filter(_rodzic).count();
			if (rodzic != 1) return false;
			int dzieci = dzieci(czlonkowieRodziny);
			if (rodzic + dzieci != czlonkowieRodziny.size()) return false;
			return liczbaDzieci.test(dzieci);
		};
	}
	
	private Predicate<SwiadczenieSprawozdawcze> dowolnySklad(IntPredicate liczbaOsob) {
		return swiadczenieSprawozdawcze -> {
			int liczbaCzlonkow = swiadczenieSprawozdawcze.getCzlonkowieRodziny().size();
			return liczbaOsob.test(liczbaCzlonkow);
		};
	}
	
	// predykaty słownikowe
	
	private Predicate<CzlonekRodziny> plec(String kod) {
		return czlonekRodziny -> kod.equals(czlonekRodziny.getKodPlci());
	}
	
	private Predicate<CzlonekRodziny> stanCywilny(String kod) {
		return czlonekRodziny ->  kod.equals(czlonekRodziny.getSytuacjaOsoby().getKodStanuCywilnego());
	}
	
	private Predicate<CzlonekRodziny> stopienPokrewienstwa(String kod) {
		return czlonekRodziny ->  kod.equals(czlonekRodziny.getSytuacjaOsoby().getKodPokrewienstwaZGlowaRodziny());
	}
	
	// predykaty słownikowe c.d.
	
	private Predicate<CzlonekRodziny> MEZCZYZNA = plec("1");
	private Predicate<CzlonekRodziny> KOBIETA = plec("2");
	
	private Predicate<CzlonekRodziny> KAWALER = stanCywilny("01");
	private Predicate<CzlonekRodziny> PANNA = stanCywilny("02");
	private Predicate<CzlonekRodziny> ZONATY = stanCywilny("03");
	private Predicate<CzlonekRodziny> ZAMEZNA = stanCywilny("04");
	private Predicate<CzlonekRodziny> KONKUBENT = stanCywilny("09");
	private Predicate<CzlonekRodziny> KONKUBINA = stanCywilny("10");
	
	private Predicate<CzlonekRodziny> MAZ = stopienPokrewienstwa("15");
	private Predicate<CzlonekRodziny> ZONA = stopienPokrewienstwa("16");
	private Predicate<CzlonekRodziny> SYN = stopienPokrewienstwa("03");
	private Predicate<CzlonekRodziny> CORKA = stopienPokrewienstwa("04");
	private Predicate<CzlonekRodziny> DZIECKO_OBCE = stopienPokrewienstwa("24");
	
	private Predicate<CzlonekRodziny> POKREWIENSTWO_NIEOKRESLONE = stopienPokrewienstwa("98");
	private Predicate<CzlonekRodziny> BRAK_POKREWIENSTWA = stopienPokrewienstwa("99");
	
	// metody pomocnicze
	
	private int dzieci(List<CzlonekRodziny> czlonkowieRodziny) {
		int synowie = (int) czlonkowieRodziny.stream().filter(MEZCZYZNA.and(KAWALER).and(SYN)).count();
		int corki = (int) czlonkowieRodziny.stream().filter(KOBIETA.and(PANNA).and(CORKA)).count();
		int obce = (int) czlonkowieRodziny.stream().filter(KAWALER.or(PANNA).and(DZIECKO_OBCE)).count();
		return synowie + corki + obce;
	}
}
